﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/ec2/EC2Request.h>
#include <aws/ec2/EC2_EXPORTS.h>
#include <aws/ec2/model/Filter.h>

#include <utility>

namespace Aws {
namespace EC2 {
namespace Model {

/**
 */
class DescribeVpcEndpointAssociationsRequest : public EC2Request {
 public:
  AWS_EC2_API DescribeVpcEndpointAssociationsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DescribeVpcEndpointAssociations"; }

  AWS_EC2_API Aws::String SerializePayload() const override;

 protected:
  AWS_EC2_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>Checks whether you have the required permissions for the action, without
   * actually making the request, and provides an error response. If you have the
   * required permissions, the error response is <code>DryRunOperation</code>.
   * Otherwise, it is <code>UnauthorizedOperation</code>.</p>
   */
  inline bool GetDryRun() const { return m_dryRun; }
  inline bool DryRunHasBeenSet() const { return m_dryRunHasBeenSet; }
  inline void SetDryRun(bool value) {
    m_dryRunHasBeenSet = true;
    m_dryRun = value;
  }
  inline DescribeVpcEndpointAssociationsRequest& WithDryRun(bool value) {
    SetDryRun(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The IDs of the VPC endpoints.</p>
   */
  inline const Aws::Vector<Aws::String>& GetVpcEndpointIds() const { return m_vpcEndpointIds; }
  inline bool VpcEndpointIdsHasBeenSet() const { return m_vpcEndpointIdsHasBeenSet; }
  template <typename VpcEndpointIdsT = Aws::Vector<Aws::String>>
  void SetVpcEndpointIds(VpcEndpointIdsT&& value) {
    m_vpcEndpointIdsHasBeenSet = true;
    m_vpcEndpointIds = std::forward<VpcEndpointIdsT>(value);
  }
  template <typename VpcEndpointIdsT = Aws::Vector<Aws::String>>
  DescribeVpcEndpointAssociationsRequest& WithVpcEndpointIds(VpcEndpointIdsT&& value) {
    SetVpcEndpointIds(std::forward<VpcEndpointIdsT>(value));
    return *this;
  }
  template <typename VpcEndpointIdsT = Aws::String>
  DescribeVpcEndpointAssociationsRequest& AddVpcEndpointIds(VpcEndpointIdsT&& value) {
    m_vpcEndpointIdsHasBeenSet = true;
    m_vpcEndpointIds.emplace_back(std::forward<VpcEndpointIdsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The filters.</p> <ul> <li> <p> <code>vpc-endpoint-id</code> - The ID of the
   * VPC endpoint.</p> </li> <li> <p> <code>associated-resource-accessibility</code>
   * - The association state. When the state is <code>accessible</code>, it returns
   * <code>AVAILABLE</code>. When the state is <code>inaccessible</code>, it returns
   * <code>PENDING</code> or <code>FAILED</code>.</p> </li> <li> <p>
   * <code>association-id</code> - The ID of the VPC endpoint association.</p> </li>
   * <li> <p> <code>associated-resource-id</code> - The ID of the associated resource
   * configuration.</p> </li> <li> <p> <code>service-network-arn</code> - The Amazon
   * Resource Name (ARN) of the associated service network. Only VPC endpoints of
   * type service network will be returned.</p> </li> <li> <p>
   * <code>resource-configuration-group-arn</code> - The Amazon Resource Name (ARN)
   * of the resource configuration of type GROUP.</p> </li> </ul>
   */
  inline const Aws::Vector<Filter>& GetFilters() const { return m_filters; }
  inline bool FiltersHasBeenSet() const { return m_filtersHasBeenSet; }
  template <typename FiltersT = Aws::Vector<Filter>>
  void SetFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters = std::forward<FiltersT>(value);
  }
  template <typename FiltersT = Aws::Vector<Filter>>
  DescribeVpcEndpointAssociationsRequest& WithFilters(FiltersT&& value) {
    SetFilters(std::forward<FiltersT>(value));
    return *this;
  }
  template <typename FiltersT = Filter>
  DescribeVpcEndpointAssociationsRequest& AddFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters.emplace_back(std::forward<FiltersT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum page size.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline DescribeVpcEndpointAssociationsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The pagination token.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  DescribeVpcEndpointAssociationsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}
 private:
  bool m_dryRun{false};

  Aws::Vector<Aws::String> m_vpcEndpointIds;

  Aws::Vector<Filter> m_filters;

  int m_maxResults{0};

  Aws::String m_nextToken;
  bool m_dryRunHasBeenSet = false;
  bool m_vpcEndpointIdsHasBeenSet = false;
  bool m_filtersHasBeenSet = false;
  bool m_maxResultsHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace EC2
}  // namespace Aws
