﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/groundstation/GroundStationRequest.h>
#include <aws/groundstation/GroundStation_EXPORTS.h>
#include <aws/groundstation/model/ContactStatus.h>
#include <aws/groundstation/model/EphemerisFilter.h>

#include <utility>

namespace Aws {
namespace GroundStation {
namespace Model {

/**
 * <p/><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/ListContactsRequest">AWS
 * API Reference</a></p>
 */
class ListContactsRequest : public GroundStationRequest {
 public:
  AWS_GROUNDSTATION_API ListContactsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListContacts"; }

  AWS_GROUNDSTATION_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>Maximum number of contacts returned.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListContactsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Next token returned in the request of a previous <code>ListContacts</code>
   * call. Used to get the next page of results.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListContactsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Status of a contact reservation.</p>
   */
  inline const Aws::Vector<ContactStatus>& GetStatusList() const { return m_statusList; }
  inline bool StatusListHasBeenSet() const { return m_statusListHasBeenSet; }
  template <typename StatusListT = Aws::Vector<ContactStatus>>
  void SetStatusList(StatusListT&& value) {
    m_statusListHasBeenSet = true;
    m_statusList = std::forward<StatusListT>(value);
  }
  template <typename StatusListT = Aws::Vector<ContactStatus>>
  ListContactsRequest& WithStatusList(StatusListT&& value) {
    SetStatusList(std::forward<StatusListT>(value));
    return *this;
  }
  inline ListContactsRequest& AddStatusList(ContactStatus value) {
    m_statusListHasBeenSet = true;
    m_statusList.push_back(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Start time of a contact in UTC.</p>
   */
  inline const Aws::Utils::DateTime& GetStartTime() const { return m_startTime; }
  inline bool StartTimeHasBeenSet() const { return m_startTimeHasBeenSet; }
  template <typename StartTimeT = Aws::Utils::DateTime>
  void SetStartTime(StartTimeT&& value) {
    m_startTimeHasBeenSet = true;
    m_startTime = std::forward<StartTimeT>(value);
  }
  template <typename StartTimeT = Aws::Utils::DateTime>
  ListContactsRequest& WithStartTime(StartTimeT&& value) {
    SetStartTime(std::forward<StartTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>End time of a contact in UTC.</p>
   */
  inline const Aws::Utils::DateTime& GetEndTime() const { return m_endTime; }
  inline bool EndTimeHasBeenSet() const { return m_endTimeHasBeenSet; }
  template <typename EndTimeT = Aws::Utils::DateTime>
  void SetEndTime(EndTimeT&& value) {
    m_endTimeHasBeenSet = true;
    m_endTime = std::forward<EndTimeT>(value);
  }
  template <typename EndTimeT = Aws::Utils::DateTime>
  ListContactsRequest& WithEndTime(EndTimeT&& value) {
    SetEndTime(std::forward<EndTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Name of a ground station.</p>
   */
  inline const Aws::String& GetGroundStation() const { return m_groundStation; }
  inline bool GroundStationHasBeenSet() const { return m_groundStationHasBeenSet; }
  template <typename GroundStationT = Aws::String>
  void SetGroundStation(GroundStationT&& value) {
    m_groundStationHasBeenSet = true;
    m_groundStation = std::forward<GroundStationT>(value);
  }
  template <typename GroundStationT = Aws::String>
  ListContactsRequest& WithGroundStation(GroundStationT&& value) {
    SetGroundStation(std::forward<GroundStationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>ARN of a satellite.</p>
   */
  inline const Aws::String& GetSatelliteArn() const { return m_satelliteArn; }
  inline bool SatelliteArnHasBeenSet() const { return m_satelliteArnHasBeenSet; }
  template <typename SatelliteArnT = Aws::String>
  void SetSatelliteArn(SatelliteArnT&& value) {
    m_satelliteArnHasBeenSet = true;
    m_satelliteArn = std::forward<SatelliteArnT>(value);
  }
  template <typename SatelliteArnT = Aws::String>
  ListContactsRequest& WithSatelliteArn(SatelliteArnT&& value) {
    SetSatelliteArn(std::forward<SatelliteArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>ARN of a mission profile.</p>
   */
  inline const Aws::String& GetMissionProfileArn() const { return m_missionProfileArn; }
  inline bool MissionProfileArnHasBeenSet() const { return m_missionProfileArnHasBeenSet; }
  template <typename MissionProfileArnT = Aws::String>
  void SetMissionProfileArn(MissionProfileArnT&& value) {
    m_missionProfileArnHasBeenSet = true;
    m_missionProfileArn = std::forward<MissionProfileArnT>(value);
  }
  template <typename MissionProfileArnT = Aws::String>
  ListContactsRequest& WithMissionProfileArn(MissionProfileArnT&& value) {
    SetMissionProfileArn(std::forward<MissionProfileArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Filter for selecting contacts that use a specific ephemeris".</p>
   */
  inline const EphemerisFilter& GetEphemeris() const { return m_ephemeris; }
  inline bool EphemerisHasBeenSet() const { return m_ephemerisHasBeenSet; }
  template <typename EphemerisT = EphemerisFilter>
  void SetEphemeris(EphemerisT&& value) {
    m_ephemerisHasBeenSet = true;
    m_ephemeris = std::forward<EphemerisT>(value);
  }
  template <typename EphemerisT = EphemerisFilter>
  ListContactsRequest& WithEphemeris(EphemerisT&& value) {
    SetEphemeris(std::forward<EphemerisT>(value));
    return *this;
  }
  ///@}
 private:
  int m_maxResults{0};

  Aws::String m_nextToken;

  Aws::Vector<ContactStatus> m_statusList;

  Aws::Utils::DateTime m_startTime{};

  Aws::Utils::DateTime m_endTime{};

  Aws::String m_groundStation;

  Aws::String m_satelliteArn;

  Aws::String m_missionProfileArn;

  EphemerisFilter m_ephemeris;
  bool m_maxResultsHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
  bool m_statusListHasBeenSet = false;
  bool m_startTimeHasBeenSet = false;
  bool m_endTimeHasBeenSet = false;
  bool m_groundStationHasBeenSet = false;
  bool m_satelliteArnHasBeenSet = false;
  bool m_missionProfileArnHasBeenSet = false;
  bool m_ephemerisHasBeenSet = false;
};

}  // namespace Model
}  // namespace GroundStation
}  // namespace Aws
