﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/inspector2/Inspector2Request.h>
#include <aws/inspector2/Inspector2_EXPORTS.h>
#include <aws/inspector2/model/CisSecurityLevel.h>
#include <aws/inspector2/model/CreateCisTargets.h>
#include <aws/inspector2/model/Schedule.h>

#include <utility>

namespace Aws {
namespace Inspector2 {
namespace Model {

/**
 */
class CreateCisScanConfigurationRequest : public Inspector2Request {
 public:
  AWS_INSPECTOR2_API CreateCisScanConfigurationRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateCisScanConfiguration"; }

  AWS_INSPECTOR2_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The scan name for the CIS scan configuration.</p>
   */
  inline const Aws::String& GetScanName() const { return m_scanName; }
  inline bool ScanNameHasBeenSet() const { return m_scanNameHasBeenSet; }
  template <typename ScanNameT = Aws::String>
  void SetScanName(ScanNameT&& value) {
    m_scanNameHasBeenSet = true;
    m_scanName = std::forward<ScanNameT>(value);
  }
  template <typename ScanNameT = Aws::String>
  CreateCisScanConfigurationRequest& WithScanName(ScanNameT&& value) {
    SetScanName(std::forward<ScanNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The security level for the CIS scan configuration. Security level refers to
   * the Benchmark levels that CIS assigns to a profile. </p>
   */
  inline CisSecurityLevel GetSecurityLevel() const { return m_securityLevel; }
  inline bool SecurityLevelHasBeenSet() const { return m_securityLevelHasBeenSet; }
  inline void SetSecurityLevel(CisSecurityLevel value) {
    m_securityLevelHasBeenSet = true;
    m_securityLevel = value;
  }
  inline CreateCisScanConfigurationRequest& WithSecurityLevel(CisSecurityLevel value) {
    SetSecurityLevel(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The schedule for the CIS scan configuration.</p>
   */
  inline const Schedule& GetSchedule() const { return m_schedule; }
  inline bool ScheduleHasBeenSet() const { return m_scheduleHasBeenSet; }
  template <typename ScheduleT = Schedule>
  void SetSchedule(ScheduleT&& value) {
    m_scheduleHasBeenSet = true;
    m_schedule = std::forward<ScheduleT>(value);
  }
  template <typename ScheduleT = Schedule>
  CreateCisScanConfigurationRequest& WithSchedule(ScheduleT&& value) {
    SetSchedule(std::forward<ScheduleT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The targets for the CIS scan configuration.</p>
   */
  inline const CreateCisTargets& GetTargets() const { return m_targets; }
  inline bool TargetsHasBeenSet() const { return m_targetsHasBeenSet; }
  template <typename TargetsT = CreateCisTargets>
  void SetTargets(TargetsT&& value) {
    m_targetsHasBeenSet = true;
    m_targets = std::forward<TargetsT>(value);
  }
  template <typename TargetsT = CreateCisTargets>
  CreateCisScanConfigurationRequest& WithTargets(TargetsT&& value) {
    SetTargets(std::forward<TargetsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The tags for the CIS scan configuration.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  CreateCisScanConfigurationRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  CreateCisScanConfigurationRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_scanName;
  bool m_scanNameHasBeenSet = false;

  CisSecurityLevel m_securityLevel{CisSecurityLevel::NOT_SET};
  bool m_securityLevelHasBeenSet = false;

  Schedule m_schedule;
  bool m_scheduleHasBeenSet = false;

  CreateCisTargets m_targets;
  bool m_targetsHasBeenSet = false;

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace Inspector2
}  // namespace Aws
