from __future__ import annotations
import collections.abc
import datetime
import typing
from warnings import deprecated # type: ignore

import jpype # type: ignore
import jpype.protocol # type: ignore

import docking.action
import docking.widgets.table
import docking.widgets.table.threaded
import ghidra.app.nav
import ghidra.app.services
import ghidra.framework.plugintool
import ghidra.program.model.address
import ghidra.program.model.listing
import ghidra.program.model.symbol
import ghidra.program.util
import ghidra.util.datastruct
import ghidra.util.table.column
import ghidra.util.table.field
import ghidra.util.task
import java.awt # type: ignore
import java.lang # type: ignore
import java.lang.ref # type: ignore
import java.util # type: ignore
import java.util.function # type: ignore
import javax.swing # type: ignore
import javax.swing.event # type: ignore
import javax.swing.table # type: ignore


COLUMN_TYPE = typing.TypeVar("COLUMN_TYPE")
EXPECTED_ROW_TYPE = typing.TypeVar("EXPECTED_ROW_TYPE")
ROW_OBJECT = typing.TypeVar("ROW_OBJECT")
ROW_TYPE = typing.TypeVar("ROW_TYPE")
T = typing.TypeVar("T")


class ProgramTableModel(java.lang.Object):
    """
    An interface for translating table rows and columns into program locations and selections.
    """

    class_: typing.ClassVar[java.lang.Class]

    def getProgram(self) -> ghidra.program.model.listing.Program:
        """
        Returns the program associated with this ProgramTableModel.
        
        :return: the program associated with this ProgramTableModel.
        :rtype: ghidra.program.model.listing.Program
        """

    def getProgramLocation(self, modelRow: typing.Union[jpype.JInt, int], modelColumn: typing.Union[jpype.JInt, int]) -> ghidra.program.util.ProgramLocation:
        """
        Returns a program location corresponding the given row and column.
         
        
        Motivation: Given a table that has a column that contains addresses. If the user clicks on
        this column, then it would be nice to have the CodeBrowser navigate to this address.
        
        :param jpype.JInt or int modelRow: the row
        :param jpype.JInt or int modelColumn: the column in the model's index
        :return: a program location corresponding the given row and column
        :rtype: ghidra.program.util.ProgramLocation
        """

    def getProgramSelection(self, modelRows: jpype.JArray[jpype.JInt]) -> ghidra.program.util.ProgramSelection:
        """
        Returns a program selection corresponding to the specified row index array. This array will
        contain the currently selected rows.
        
        :param jpype.JArray[jpype.JInt] modelRows: the currently selected rows.
        :return: a program selection
        :rtype: ghidra.program.util.ProgramSelection
        """

    @property
    def programSelection(self) -> ghidra.program.util.ProgramSelection:
        ...

    @property
    def program(self) -> ghidra.program.model.listing.Program:
        ...


class PreviewDataTableCellRenderer(ghidra.util.table.column.AbstractGColumnRenderer[PreviewTableCellData]):
    """
    A custom renderer used to display what is at the :obj:`ProgramLocation` similarly to
    how it is displayed in the Listing window..  This class is meant to be
    used directly with :obj:`PreviewTableCellData` column data.
    """

    class_: typing.ClassVar[java.lang.Class]

    def __init__(self):
        ...


class EmptyThreadedTableModel(docking.widgets.table.threaded.ThreadedTableModelStub[T], typing.Generic[T]):
    """
    An empty implementation of the ThreadedTableModel.
    """

    @typing.type_check_only
    class NamedEmptyTableColumn(docking.widgets.table.AbstractDynamicTableColumnStub[T, java.lang.String]):
        ...
        class_: typing.ClassVar[java.lang.Class]


    class_: typing.ClassVar[java.lang.Class]

    def __init__(self, modelName: typing.Union[java.lang.String, str], columnNames: jpype.JArray[java.lang.String]):
        """
        Constructs a new empty table model.
        
        :param java.lang.String or str modelName: the name of the model.
        :param jpype.JArray[java.lang.String] columnNames: the column names.
        """

    def getProgram(self) -> ghidra.program.model.listing.Program:
        ...

    def getProgramLocation(self, row: typing.Union[jpype.JInt, int], column: typing.Union[jpype.JInt, int]) -> ghidra.program.util.ProgramLocation:
        ...

    def getProgramSelection(self, rows: jpype.JArray[jpype.JInt]) -> ghidra.program.util.ProgramSelection:
        ...

    @property
    def programSelection(self) -> ghidra.program.util.ProgramSelection:
        ...

    @property
    def program(self) -> ghidra.program.model.listing.Program:
        ...


class GhidraTableCellRenderer(docking.widgets.table.GTableCellRenderer):

    class_: typing.ClassVar[java.lang.Class]

    @typing.overload
    def __init__(self):
        ...

    @typing.overload
    def __init__(self, f: java.awt.Font):
        """
        Constructs a new GhidraTableCellRenderer using the specified font.
        
        :param java.awt.Font f: the font to use when rendering text in the table cells
        """


class CompositeGhidraTableCellRenderer(GhidraTableCellRenderer):
    """
    What: A cell renderer that will attempt to use any registered cell renderer and will otherwise
        default to the parent rendering implementation.
    Why:  Sometimes the need arises to be able to use the default table rendering while adding 
        additional rendering (e.g., to be able to add row coloring).
    How:  Create a cell renderer that extends this class and install that into your table.  Then,
        override :meth:`getTableCellRendererComponent(JTable, Object, boolean, boolean, int, int) <.getTableCellRendererComponent>`
        to call this class' implementation.  Finally, add desired decoration.
    """

    class_: typing.ClassVar[java.lang.Class]

    def __init__(self):
        ...


class PreviewTableCellData(java.lang.Comparable[PreviewTableCellData]):
    """
    A generic data type used by table models in order to signal that the data should render
    a preview for a given :obj:`ProgramLocation`, where the preview is what is displayed in 
    the Listing.
    """

    class_: typing.ClassVar[java.lang.Class]

    def __init__(self, location: ghidra.program.util.ProgramLocation, codeUnitFormat: ghidra.program.model.listing.CodeUnitFormat):
        """
        Constructor
        
        :param ghidra.program.util.ProgramLocation location: the location for the preview
        :param ghidra.program.model.listing.CodeUnitFormat codeUnitFormat: the format needed to render preview data
        """

    def getDisplayString(self) -> str:
        """
        Get the preview for the code unit at or containing the address associated with this cell's row.
        
        :return: the preview string.
        :rtype: str
        """

    def getHTMLDisplayString(self) -> str:
        """
        Get the preview as HTML for the code unit at or containing the address associated with this cell's row.
        
        :return: the preview string.
        :rtype: str
        """

    def getProgramLocation(self) -> ghidra.program.util.ProgramLocation:
        ...

    def isOffcut(self) -> bool:
        ...

    @property
    def hTMLDisplayString(self) -> java.lang.String:
        ...

    @property
    def offcut(self) -> jpype.JBoolean:
        ...

    @property
    def displayString(self) -> java.lang.String:
        ...

    @property
    def programLocation(self) -> ghidra.program.util.ProgramLocation:
        ...


class IncomingReferencesTableModel(AddressBasedTableModel[ghidra.util.table.field.ReferenceEndpoint]):

    class_: typing.ClassVar[java.lang.Class]

    def __init__(self, title: typing.Union[java.lang.String, str], serviceProvider: ghidra.framework.plugintool.ServiceProvider, program: ghidra.program.model.listing.Program, references: java.util.List[ghidra.util.table.field.OutgoingReferenceEndpoint], monitor: ghidra.util.task.TaskMonitor):
        ...


class GhidraTable(docking.widgets.table.GTable):
    """
    Navigation is only supported if the underlying table model implements :obj:`ProgramTableModel`
    and the ``setGoToService()`` method has been called with a valid reference. When both of
    these conditions are met, then the table will navigate on a user's double-click or on an
    ``Enter`` key press. Also, if selection navigation is enabled, then this table will
    navigate **any time the selection of the table changes**. To prevent this feature call
    :meth:`setNavigateOnSelectionEnabled(boolean) <.setNavigateOnSelectionEnabled>` with a value of false.
    """

    @typing.type_check_only
    class SelectionListener(javax.swing.event.ListSelectionListener):
        ...
        class_: typing.ClassVar[java.lang.Class]


    @typing.type_check_only
    class GoToServiceProvider(ghidra.framework.plugintool.ServiceProviderStub):
        ...
        class_: typing.ClassVar[java.lang.Class]


    class_: typing.ClassVar[java.lang.Class]

    @typing.overload
    def __init__(self):
        ...

    @typing.overload
    def __init__(self, model: javax.swing.table.TableModel):
        ...

    def getProgram(self) -> ghidra.program.model.listing.Program:
        """
        Returns the program being used by this table; null if the underlying model does not implement
        :obj:`ProgramTableModel`
        
        :return: the table's program
        :rtype: ghidra.program.model.listing.Program
        """

    def getProgramSelection(self) -> ghidra.program.util.ProgramSelection:
        """
        Returns the program selection equivalent to the rows currently selected in the table.
         
        
        This method is only valid when the underlying table model implements
        :obj:`ProgramTableModel`.
         
        
        Returns null if no rows are selected or the underlying model does not implement
        ``ProgramTableModel``.
        
        :return: the program selection or null.
        :rtype: ghidra.program.util.ProgramSelection
        """

    @typing.overload
    @deprecated("use installNavigation(ServiceProvider) or\n             installNavigation(ServiceProvider,Navigatable)")
    def installNavigation(self, goToService: ghidra.app.services.GoToService, nav: ghidra.app.nav.Navigatable):
        """
        Sets the GoTo service to use when navigation is enabled on this table.
        
        :param ghidra.app.services.GoToService goToService: the GoTo service
        :param ghidra.app.nav.Navigatable nav: the navigable
        
        .. deprecated::
        
        use :meth:`installNavigation(ServiceProvider) <.installNavigation>` or
                    :meth:`installNavigation(ServiceProvider,Navigatable) <.installNavigation>`
        """

    @typing.overload
    def installNavigation(self, sp: ghidra.framework.plugintool.ServiceProvider, nav: ghidra.app.nav.Navigatable):
        """
        Sets the service provider to use when navigation is enabled on this table.
         
        
        The service provider will be used to retrieve the :obj:`GoToService`, as needed after the
        system has been initialized. If you do not have a :obj:`Navigatable` preferences, then call
        :meth:`installNavigation(ServiceProvider) <.installNavigation>` instead.
        
        :param ghidra.framework.plugintool.ServiceProvider sp: the service provider
        :param ghidra.app.nav.Navigatable nav: the navigable
        """

    @typing.overload
    def installNavigation(self, sp: ghidra.framework.plugintool.ServiceProvider):
        """
        Sets the service provider to use when navigation is enabled on this table.
         
        
        The service provider will be used to retrieve the :obj:`GoToService`, as needed after the
        system has been initialized.
        
        :param ghidra.framework.plugintool.ServiceProvider sp: the service provider
        """

    def navigate(self, row: typing.Union[jpype.JInt, int], column: typing.Union[jpype.JInt, int]):
        """
        Navigate to the program location denoted by the given row and column
         
        
        Does nothing if no :obj:`GoToService` has been installed from
        :meth:`installNavigation(GoToService, Navigatable) <.installNavigation>`. Also, this method will do nothing if
        this table's ``TableModel`` is not an instance of :obj:`ProgramTableModel`.
        
        :param jpype.JInt or int row: the row
        :param jpype.JInt or int column: the column
        """

    def removeNavigation(self):
        """
        Removes any installed navigation components, such as listeners, a navigatable and the service
        provider.
        """

    def selectRow(self, row: typing.Union[jpype.JInt, int]):
        """
        Selects the given row and performs a goto, if applicable.
        
        :param jpype.JInt or int row: The row to select
        """

    def setNavigateOnSelectionEnabled(self, enabled: typing.Union[jpype.JBoolean, bool]):
        """
        Allows the user to enable and disable the table's feature that triggers navigation on certain
        selection events, like mouse clicking and pressing the 'Enter' key.
        
        :param jpype.JBoolean or bool enabled: true enables the navigation on selection feature.
        """

    @property
    def programSelection(self) -> ghidra.program.util.ProgramSelection:
        ...

    @property
    def program(self) -> ghidra.program.model.listing.Program:
        ...


class GhidraThreadedTablePanel(docking.widgets.table.threaded.GThreadedTablePanel[T], typing.Generic[T]):

    class_: typing.ClassVar[java.lang.Class]

    @typing.overload
    def __init__(self, model: docking.widgets.table.threaded.ThreadedTableModel[T, typing.Any]):
        ...

    @typing.overload
    def __init__(self, model: docking.widgets.table.threaded.ThreadedTableModel[T, typing.Any], minUpdateDelay: typing.Union[jpype.JInt, int]):
        ...

    @typing.overload
    def __init__(self, model: docking.widgets.table.threaded.ThreadedTableModel[T, typing.Any], minUpdateDelay: typing.Union[jpype.JInt, int], maxUpdateDelay: typing.Union[jpype.JInt, int]):
        ...


class CustomLoadingAddressTableModel(AddressPreviewTableModel):
    """
    An :obj:`Address` based table model that allows clients to load their data via 
    the :obj:`TableModelLoader` callback provided at construction time.
     
    
    Why?  Well, this allows clients to use the existing table model framework without
    having to create a new table model.  In other words, some of the boilerplate code
    of creating a model is removed--clients need only implement one method in order to
    get full thread table functionality, which is a lot.
    """

    class_: typing.ClassVar[java.lang.Class]

    @typing.overload
    def __init__(self, modelName: typing.Union[java.lang.String, str], serviceProvider: ghidra.framework.plugintool.ServiceProvider, program: ghidra.program.model.listing.Program, loader: TableModelLoader[ghidra.program.model.address.Address], monitor: ghidra.util.task.TaskMonitor):
        ...

    @typing.overload
    def __init__(self, modelName: typing.Union[java.lang.String, str], serviceProvider: ghidra.framework.plugintool.ServiceProvider, program: ghidra.program.model.listing.Program, loader: TableModelLoader[ghidra.program.model.address.Address], monitor: ghidra.util.task.TaskMonitor, loadIncrementally: typing.Union[jpype.JBoolean, bool]):
        ...


class CodeUnitTableCellRenderer(ghidra.util.table.column.AbstractGColumnRenderer[ghidra.util.table.field.CodeUnitTableCellData]):
    """
    Renderer for :obj:`CodeUnitTableCellData`s
    """

    class_: typing.ClassVar[java.lang.Class]

    def __init__(self):
        ...


class AbstractSelectionNavigationAction(docking.action.ToggleDockingAction):
    """
    
    .. _description:
    
    
    An action used to trigger navigation callback on instances of :obj:`JTable`.  Users can 
    toggle this action to control navigation that is based upon selection.
     
    
    Subclasses need to implement :meth:`navigate() <.navigate>`, which will be called when a navigation is
    triggered on the given table by a selection.
     
    
    This class will save the state of the action when the tool is saved.
    """

    @typing.type_check_only
    class SelectionListener(javax.swing.event.ListSelectionListener):
        ...
        class_: typing.ClassVar[java.lang.Class]


    class_: typing.ClassVar[java.lang.Class]

    def navigate(self):
        """
        Users of this class will implement this method to know when to use their table to perform
        navigation tasks in their own way.
        """


class TableModelLoader(java.lang.Object, typing.Generic[T]):
    """
    Allows clients to create a table model that will call them back via this interface so
    that they may perform their own loading.
    """

    class_: typing.ClassVar[java.lang.Class]

    def load(self, accumulator: ghidra.util.datastruct.Accumulator[T], monitor: ghidra.util.task.TaskMonitor):
        ...


class ProgramMappedTableColumn(docking.widgets.table.MappedTableColumn[ROW_TYPE, EXPECTED_ROW_TYPE, COLUMN_TYPE, ghidra.program.model.listing.Program], typing.Generic[ROW_TYPE, EXPECTED_ROW_TYPE, COLUMN_TYPE]):
    ...
    class_: typing.ClassVar[java.lang.Class]


class MappedProgramLocationTableColumn(docking.widgets.table.MappedTableColumn[ROW_TYPE, EXPECTED_ROW_TYPE, COLUMN_TYPE, ghidra.program.model.listing.Program], ghidra.util.table.field.ProgramLocationTableColumn[ROW_TYPE, COLUMN_TYPE], typing.Generic[ROW_TYPE, EXPECTED_ROW_TYPE, COLUMN_TYPE]):
    ...
    class_: typing.ClassVar[java.lang.Class]


class SelectionNavigationAction(AbstractSelectionNavigationAction):
    """
    This action is used by :obj:`GhidraTable`s to allow the user to trigger navigation when 
    selections are made.
     
    
    This class will save the state of the action when the tool is saved.
    
    
    .. seealso::
    
        | :obj:`AbstractSelectionNavigationAction`
    """

    class_: typing.ClassVar[java.lang.Class]

    @typing.overload
    def __init__(self, plugin: ghidra.framework.plugintool.Plugin, table: GhidraTable):
        """
        Constructor that relies on an instance of :obj:`GhidraTable` to do the work of
        navigation.  Clients that have :obj:`JTable`s that are not instances of :obj:`GhidraTable`
        can use the super class action and define its :meth:`navigate() <.navigate>` callback method.
        
        :param ghidra.framework.plugintool.Plugin plugin: The owner plugin
        :param GhidraTable table: The :obj:`GhidraTable` which this action works with
        
        .. seealso::
        
            | :obj:`AbstractSelectionNavigationAction`
        """

    @typing.overload
    def __init__(self, owner: typing.Union[java.lang.String, str], table: GhidraTable):
        """
        Constructor that relies on an instance of :obj:`GhidraTable` to do the work of
        navigation.  Clients that have :obj:`JTable`s that are not instances of :obj:`GhidraTable`
        can use the super class action and define its :meth:`navigate() <.navigate>` callback method.
        
        :param java.lang.String or str owner: The owner name
        :param GhidraTable table: The :obj:`GhidraTable` which this action works with
        
        .. seealso::
        
            | :obj:`AbstractSelectionNavigationAction`
        """


class GhidraTableFilterPanel(docking.widgets.table.GTableFilterPanel[ROW_OBJECT], typing.Generic[ROW_OBJECT]):
    """
    This is a "program aware" version of GTableFilterPanel
    """

    class_: typing.ClassVar[java.lang.Class]

    @typing.overload
    def __init__(self, table: javax.swing.JTable, tableModel: docking.widgets.table.RowObjectTableModel[ROW_OBJECT]):
        """
        Creates a table filter panel that filters the contents of the given table.
        
        :param javax.swing.JTable table: The table whose contents will be filtered.
        :param docking.widgets.table.RowObjectTableModel[ROW_OBJECT] tableModel: The table model used by the table--passed in by the type that we require
        """

    @typing.overload
    def __init__(self, table: javax.swing.JTable, tableModel: docking.widgets.table.RowObjectTableModel[ROW_OBJECT], filterLabel: typing.Union[java.lang.String, str]):
        ...


class AddressArrayTableModel(AddressPreviewTableModel):
    """
    This table model of addresses is used when you already have an
    Address array built. 
     
    
    If you need to compute the address array, then you should
    extend :obj:`AddressPreviewTableModel` and override the 
    :meth:`doLoad(Accumulator<Address> accumulator, TaskMonitor monitor) <AddressPreviewTableModel.doLoad>`
    method which will be called in a dedicated thread.
      
    
    Alternatively, you can create an instance of the :obj:`CustomLoadingAddressTableModel`,
    supplying your own loading via the :obj:`TableModelLoader`.
    """

    class_: typing.ClassVar[java.lang.Class]

    @typing.overload
    def __init__(self, title: typing.Union[java.lang.String, str], serviceProvider: ghidra.framework.plugintool.ServiceProvider, prog: ghidra.program.model.listing.Program, addrs: jpype.JArray[ghidra.program.model.address.Address], monitor: ghidra.util.task.TaskMonitor):
        """
        Constructor.
        
        :param java.lang.String or str title: title of the query
        :param ghidra.framework.plugintool.ServiceProvider serviceProvider: from which to get services
        :param ghidra.program.model.listing.Program prog: program
        :param jpype.JArray[ghidra.program.model.address.Address] addrs: array of addresses in the model
        :param ghidra.util.task.TaskMonitor monitor: monitor that is used to show progress; may be null
        """

    @typing.overload
    def __init__(self, title: typing.Union[java.lang.String, str], serviceProvider: ghidra.framework.plugintool.ServiceProvider, prog: ghidra.program.model.listing.Program, addrs: jpype.JArray[ghidra.program.model.address.Address]):
        """
        Constructor.
        
        :param java.lang.String or str title: title of the query
        :param ghidra.framework.plugintool.ServiceProvider serviceProvider: from which to get services
        :param ghidra.program.model.listing.Program prog: program
        :param jpype.JArray[ghidra.program.model.address.Address] addrs: array of addresses in the model
        """

    def setAddresses(self, addresses: jpype.JArray[ghidra.program.model.address.Address]):
        ...


class AddressBasedTableModel(GhidraProgramTableModel[ROW_TYPE], typing.Generic[ROW_TYPE]):
    """
    This class is now just a shell left in place to not break external clients.
    """

    class_: typing.ClassVar[java.lang.Class]

    @typing.overload
    def __init__(self, title: typing.Union[java.lang.String, str], serviceProvider: ghidra.framework.plugintool.ServiceProvider, program: ghidra.program.model.listing.Program, monitor: ghidra.util.task.TaskMonitor):
        ...

    @typing.overload
    def __init__(self, title: typing.Union[java.lang.String, str], serviceProvider: ghidra.framework.plugintool.ServiceProvider, program: ghidra.program.model.listing.Program, monitor: ghidra.util.task.TaskMonitor, loadIncrementally: typing.Union[jpype.JBoolean, bool]):
        ...


class AddressSetTableModel(AddressPreviewTableModel):

    class_: typing.ClassVar[java.lang.Class]

    def __init__(self, title: typing.Union[java.lang.String, str], serviceProvider: ghidra.framework.plugintool.ServiceProvider, prog: ghidra.program.model.listing.Program, addressSet: ghidra.program.model.address.AddressSetView, monitor: ghidra.util.task.TaskMonitor):
        ...


class ReferencesFromTableModel(AddressBasedTableModel[ghidra.util.table.field.ReferenceEndpoint]):
    """
    Table model for showing the 'from' side of passed-in references.
    """

    @typing.type_check_only
    class ReferenceTypeTableColumn(ghidra.util.table.field.AbstractProgramBasedDynamicTableColumn[ghidra.util.table.field.ReferenceEndpoint, ghidra.util.table.field.ReferenceEndpoint]):
        ...
        class_: typing.ClassVar[java.lang.Class]


    @typing.type_check_only
    class ReferenceTypeTableCellRenderer(ghidra.util.table.column.AbstractGColumnRenderer[ghidra.util.table.field.ReferenceEndpoint]):
        ...
        class_: typing.ClassVar[java.lang.Class]


    class_: typing.ClassVar[java.lang.Class]

    @typing.overload
    def __init__(self, refs: collections.abc.Sequence, sp: ghidra.framework.plugintool.ServiceProvider, program: ghidra.program.model.listing.Program):
        ...

    @typing.overload
    def __init__(self, refsSupplier: java.util.function.Supplier[java.util.Collection[ghidra.program.model.symbol.Reference]], sp: ghidra.framework.plugintool.ServiceProvider, program: ghidra.program.model.listing.Program):
        ...


class GhidraFilterTable(docking.widgets.table.GFilterTable[ROW_OBJECT], typing.Generic[ROW_OBJECT]):

    class_: typing.ClassVar[java.lang.Class]

    def __init__(self, model: docking.widgets.table.RowObjectTableModel[ROW_OBJECT]):
        ...

    @typing.overload
    def installNavigation(self, goToService: ghidra.app.services.GoToService, nav: ghidra.app.nav.Navigatable):
        ...

    @typing.overload
    def installNavigation(self, goToService: ghidra.app.services.GoToService):
        ...

    @typing.overload
    def installNavigation(self, provider: ghidra.framework.plugintool.ServiceProvider):
        ...

    def removeNavigation(self):
        ...

    def setNavigateOnSelectionEnabled(self, b: typing.Union[jpype.JBoolean, bool]):
        ...


class ProgramLocationTableRowMapper(docking.widgets.table.TableRowMapper[ROW_TYPE, EXPECTED_ROW_TYPE, ghidra.program.model.listing.Program], typing.Generic[ROW_TYPE, EXPECTED_ROW_TYPE]):
    """
    NOTE:  ALL TableRowMapper CLASSES MUST END IN "TableRowMapper".  If not,
    the ClassSearcher will not find them.
     
    An interface that allows implementors to map an object of one type to another.  This is useful
    for table models that have row types that are easily converted to other more generic types.
    For example, the Bookmarks table model's data is based upon Bookmark objects.  Furthermore, 
    those objects are easily converted to ProgramLocations and Addresses.  By creating a mapper 
    for the these types, the table model can now show dynamic columns that work on ProgramLocations
    and Addresses.  
     
    
    This interface is an ExtensionPoint so that once created, they will be ingested automatically
    by Ghidra.  Once discovered, these mappers will be used to provide dynamic columns to
    tables with row types that match ``ROW_TYPE``.
     
    
    This column is an extension of :obj:`TableRowMapper` that has knowledge of 
    :obj:`ProgramLocationTableColumn`s, which means that it knows how to generate 
    :obj:`ProgramLocation`s.  This is the preferred mapper to use with tables that work on program
    data, as it means that the column works with navigation.
    
    
    .. seealso::
    
        | :obj:`AbstractDynamicTableColumn`
    
        | :obj:`TableUtils`
    """

    class_: typing.ClassVar[java.lang.Class]

    def __init__(self):
        ...

    def createMappedTableColumn(self, destinationColumn: docking.widgets.table.DynamicTableColumn[EXPECTED_ROW_TYPE, COLUMN_TYPE, ghidra.program.model.listing.Program]) -> docking.widgets.table.DynamicTableColumn[ROW_TYPE, COLUMN_TYPE, ghidra.program.model.listing.Program]:
        """
        Creates a table column that will create a table column that knows how to map the 
        given **ROW_TYPE** to the type of the column passed in, the **EXPECTED_ROW_TYPE**.
        
        :param COLUMN_TYPE: The column type of the given and created columns:param docking.widgets.table.DynamicTableColumn[EXPECTED_ROW_TYPE, COLUMN_TYPE, ghidra.program.model.listing.Program] destinationColumn: The existing column, which is based upon EXPECTED_ROW_TYPE,
                that we want to be able to use with the type we have, the ROW_TYPE.
        """


@typing.type_check_only
class IntObjectCache(java.lang.Object):

    @typing.type_check_only
    class MySoftRef(java.lang.ref.SoftReference[java.lang.Object]):
        ...
        class_: typing.ClassVar[java.lang.Class]


    class_: typing.ClassVar[java.lang.Class]


class AddressPreviewTableModel(AddressBasedTableModel[ghidra.program.model.address.Address]):
    """
    Table model that shows a location, label, and a preview column to
    show a preview of the code unit. The location can be in a memory address,
    a stack address, or a register address. The label is the primary symbol
    at the address, if one exists. Use this model when you have a list of
    addresses to build up dynamically.
    """

    class_: typing.ClassVar[java.lang.Class]

    def setSelectionSize(self, size: typing.Union[jpype.JInt, int]):
        """
        Sets the size of the selections generated by this model when asked to create 
        program selections.  For example, some clients know that each table row represents
        a contiguous range of 4 addresses.  In this case, when the user makes a selection, 
        that client wants the selection to be 4 addresses, starting at the address in 
        the given table row.
        
        :param jpype.JInt or int size: the size of the selections generated by this model when asked to create 
                        program selections.
        """


class GhidraProgramTableModel(docking.widgets.table.threaded.ThreadedTableModel[ROW_TYPE, ghidra.program.model.listing.Program], ProgramTableModel, typing.Generic[ROW_TYPE]):

    class_: typing.ClassVar[java.lang.Class]

    @typing.overload
    def getAddress(self, modelRow: typing.Union[jpype.JInt, int], modelColumn: typing.Union[jpype.JInt, int]) -> ghidra.program.model.address.Address:
        """
        Returns an address for the given row and column.
        
        :param jpype.JInt or int modelRow: the model row
        :param jpype.JInt or int modelColumn: the column row
        :return: the address
        :rtype: ghidra.program.model.address.Address
        """

    @typing.overload
    def getAddress(self, modelRow: typing.Union[jpype.JInt, int]) -> ghidra.program.model.address.Address:
        """
        Returns the best Address for the given row.  
         
        
        Implementation Note: this class will only return an Address if this model's row type is
        Address.  Clients that know how to get an Address for a given row should override this 
        method.
        
        :param jpype.JInt or int modelRow: the row
        :return: the Address or null
        :rtype: ghidra.program.model.address.Address
        """

    def setProgram(self, program: ghidra.program.model.listing.Program):
        ...

    @property
    def address(self) -> ghidra.program.model.address.Address:
        ...



__all__ = ["ProgramTableModel", "PreviewDataTableCellRenderer", "EmptyThreadedTableModel", "GhidraTableCellRenderer", "CompositeGhidraTableCellRenderer", "PreviewTableCellData", "IncomingReferencesTableModel", "GhidraTable", "GhidraThreadedTablePanel", "CustomLoadingAddressTableModel", "CodeUnitTableCellRenderer", "AbstractSelectionNavigationAction", "TableModelLoader", "ProgramMappedTableColumn", "MappedProgramLocationTableColumn", "SelectionNavigationAction", "GhidraTableFilterPanel", "AddressArrayTableModel", "AddressBasedTableModel", "AddressSetTableModel", "ReferencesFromTableModel", "GhidraFilterTable", "ProgramLocationTableRowMapper", "IntObjectCache", "AddressPreviewTableModel", "GhidraProgramTableModel"]
