﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/apigateway/APIGatewayRequest.h>
#include <aws/apigateway/APIGateway_EXPORTS.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace APIGateway {
namespace Model {

/**
 * <p>Requests API Gateway to create a Resource resource.</p><p><h3>See Also:</h3>
 * <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/apigateway-2015-07-09/CreateResourceRequest">AWS
 * API Reference</a></p>
 */
class CreateResourceRequest : public APIGatewayRequest {
 public:
  AWS_APIGATEWAY_API CreateResourceRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateResource"; }

  AWS_APIGATEWAY_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The string identifier of the associated RestApi.</p>
   */
  inline const Aws::String& GetRestApiId() const { return m_restApiId; }
  inline bool RestApiIdHasBeenSet() const { return m_restApiIdHasBeenSet; }
  template <typename RestApiIdT = Aws::String>
  void SetRestApiId(RestApiIdT&& value) {
    m_restApiIdHasBeenSet = true;
    m_restApiId = std::forward<RestApiIdT>(value);
  }
  template <typename RestApiIdT = Aws::String>
  CreateResourceRequest& WithRestApiId(RestApiIdT&& value) {
    SetRestApiId(std::forward<RestApiIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The parent resource's identifier.</p>
   */
  inline const Aws::String& GetParentId() const { return m_parentId; }
  inline bool ParentIdHasBeenSet() const { return m_parentIdHasBeenSet; }
  template <typename ParentIdT = Aws::String>
  void SetParentId(ParentIdT&& value) {
    m_parentIdHasBeenSet = true;
    m_parentId = std::forward<ParentIdT>(value);
  }
  template <typename ParentIdT = Aws::String>
  CreateResourceRequest& WithParentId(ParentIdT&& value) {
    SetParentId(std::forward<ParentIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The last path segment for this resource.</p>
   */
  inline const Aws::String& GetPathPart() const { return m_pathPart; }
  inline bool PathPartHasBeenSet() const { return m_pathPartHasBeenSet; }
  template <typename PathPartT = Aws::String>
  void SetPathPart(PathPartT&& value) {
    m_pathPartHasBeenSet = true;
    m_pathPart = std::forward<PathPartT>(value);
  }
  template <typename PathPartT = Aws::String>
  CreateResourceRequest& WithPathPart(PathPartT&& value) {
    SetPathPart(std::forward<PathPartT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_restApiId;

  Aws::String m_parentId;

  Aws::String m_pathPart;
  bool m_restApiIdHasBeenSet = false;
  bool m_parentIdHasBeenSet = false;
  bool m_pathPartHasBeenSet = false;
};

}  // namespace Model
}  // namespace APIGateway
}  // namespace Aws
