﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/workspaces-web/WorkSpacesWebRequest.h>
#include <aws/workspaces-web/WorkSpacesWeb_EXPORTS.h>
#include <aws/workspaces-web/model/AuthenticationType.h>
#include <aws/workspaces-web/model/InstanceType.h>
#include <aws/workspaces-web/model/Tag.h>

#include <utility>

namespace Aws {
namespace WorkSpacesWeb {
namespace Model {

/**
 */
class CreatePortalRequest : public WorkSpacesWebRequest {
 public:
  AWS_WORKSPACESWEB_API CreatePortalRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreatePortal"; }

  AWS_WORKSPACESWEB_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The name of the web portal. This is not visible to users who log into the web
   * portal.</p>
   */
  inline const Aws::String& GetDisplayName() const { return m_displayName; }
  inline bool DisplayNameHasBeenSet() const { return m_displayNameHasBeenSet; }
  template <typename DisplayNameT = Aws::String>
  void SetDisplayName(DisplayNameT&& value) {
    m_displayNameHasBeenSet = true;
    m_displayName = std::forward<DisplayNameT>(value);
  }
  template <typename DisplayNameT = Aws::String>
  CreatePortalRequest& WithDisplayName(DisplayNameT&& value) {
    SetDisplayName(std::forward<DisplayNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The tags to add to the web portal. A tag is a key-value pair.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreatePortalRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreatePortalRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The customer managed key of the web portal.</p>
   */
  inline const Aws::String& GetCustomerManagedKey() const { return m_customerManagedKey; }
  inline bool CustomerManagedKeyHasBeenSet() const { return m_customerManagedKeyHasBeenSet; }
  template <typename CustomerManagedKeyT = Aws::String>
  void SetCustomerManagedKey(CustomerManagedKeyT&& value) {
    m_customerManagedKeyHasBeenSet = true;
    m_customerManagedKey = std::forward<CustomerManagedKeyT>(value);
  }
  template <typename CustomerManagedKeyT = Aws::String>
  CreatePortalRequest& WithCustomerManagedKey(CustomerManagedKeyT&& value) {
    SetCustomerManagedKey(std::forward<CustomerManagedKeyT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The additional encryption context of the portal.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetAdditionalEncryptionContext() const { return m_additionalEncryptionContext; }
  inline bool AdditionalEncryptionContextHasBeenSet() const { return m_additionalEncryptionContextHasBeenSet; }
  template <typename AdditionalEncryptionContextT = Aws::Map<Aws::String, Aws::String>>
  void SetAdditionalEncryptionContext(AdditionalEncryptionContextT&& value) {
    m_additionalEncryptionContextHasBeenSet = true;
    m_additionalEncryptionContext = std::forward<AdditionalEncryptionContextT>(value);
  }
  template <typename AdditionalEncryptionContextT = Aws::Map<Aws::String, Aws::String>>
  CreatePortalRequest& WithAdditionalEncryptionContext(AdditionalEncryptionContextT&& value) {
    SetAdditionalEncryptionContext(std::forward<AdditionalEncryptionContextT>(value));
    return *this;
  }
  template <typename AdditionalEncryptionContextKeyT = Aws::String, typename AdditionalEncryptionContextValueT = Aws::String>
  CreatePortalRequest& AddAdditionalEncryptionContext(AdditionalEncryptionContextKeyT&& key, AdditionalEncryptionContextValueT&& value) {
    m_additionalEncryptionContextHasBeenSet = true;
    m_additionalEncryptionContext.emplace(std::forward<AdditionalEncryptionContextKeyT>(key),
                                          std::forward<AdditionalEncryptionContextValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A unique, case-sensitive identifier that you provide to ensure the
   * idempotency of the request. Idempotency ensures that an API request completes
   * only once. With an idempotent request, if the original request completes
   * successfully, subsequent retries with the same client token returns the result
   * from the original successful request. </p> <p>If you do not specify a client
   * token, one is automatically generated by the Amazon Web Services SDK.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  CreatePortalRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type of authentication integration points used when signing into the web
   * portal. Defaults to <code>Standard</code>.</p> <p> <code>Standard</code> web
   * portals are authenticated directly through your identity provider. You need to
   * call <code>CreateIdentityProvider</code> to integrate your identity provider
   * with your web portal. User and group access to your web portal is controlled
   * through your identity provider.</p> <p> <code>IAM Identity Center</code> web
   * portals are authenticated through IAM Identity Center. Identity sources
   * (including external identity provider integration), plus user and group access
   * to your web portal, can be configured in the IAM Identity Center.</p>
   */
  inline AuthenticationType GetAuthenticationType() const { return m_authenticationType; }
  inline bool AuthenticationTypeHasBeenSet() const { return m_authenticationTypeHasBeenSet; }
  inline void SetAuthenticationType(AuthenticationType value) {
    m_authenticationTypeHasBeenSet = true;
    m_authenticationType = value;
  }
  inline CreatePortalRequest& WithAuthenticationType(AuthenticationType value) {
    SetAuthenticationType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type and resources of the underlying instance.</p>
   */
  inline InstanceType GetInstanceType() const { return m_instanceType; }
  inline bool InstanceTypeHasBeenSet() const { return m_instanceTypeHasBeenSet; }
  inline void SetInstanceType(InstanceType value) {
    m_instanceTypeHasBeenSet = true;
    m_instanceType = value;
  }
  inline CreatePortalRequest& WithInstanceType(InstanceType value) {
    SetInstanceType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of concurrent sessions for the portal.</p>
   */
  inline int GetMaxConcurrentSessions() const { return m_maxConcurrentSessions; }
  inline bool MaxConcurrentSessionsHasBeenSet() const { return m_maxConcurrentSessionsHasBeenSet; }
  inline void SetMaxConcurrentSessions(int value) {
    m_maxConcurrentSessionsHasBeenSet = true;
    m_maxConcurrentSessions = value;
  }
  inline CreatePortalRequest& WithMaxConcurrentSessions(int value) {
    SetMaxConcurrentSessions(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_displayName;
  bool m_displayNameHasBeenSet = false;

  Aws::Vector<Tag> m_tags;
  bool m_tagsHasBeenSet = false;

  Aws::String m_customerManagedKey;
  bool m_customerManagedKeyHasBeenSet = false;

  Aws::Map<Aws::String, Aws::String> m_additionalEncryptionContext;
  bool m_additionalEncryptionContextHasBeenSet = false;

  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_clientTokenHasBeenSet = true;

  AuthenticationType m_authenticationType{AuthenticationType::NOT_SET};
  bool m_authenticationTypeHasBeenSet = false;

  InstanceType m_instanceType{InstanceType::NOT_SET};
  bool m_instanceTypeHasBeenSet = false;

  int m_maxConcurrentSessions{0};
  bool m_maxConcurrentSessionsHasBeenSet = false;
};

}  // namespace Model
}  // namespace WorkSpacesWeb
}  // namespace Aws
