﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/ec2/EC2Request.h>
#include <aws/ec2/EC2_EXPORTS.h>
#include <aws/ec2/model/TagSpecification.h>

#include <utility>

namespace Aws {
namespace EC2 {
namespace Model {

/**
 */
class CreateRestoreImageTaskRequest : public EC2Request {
 public:
  AWS_EC2_API CreateRestoreImageTaskRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateRestoreImageTask"; }

  AWS_EC2_API Aws::String SerializePayload() const override;

 protected:
  AWS_EC2_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The name of the Amazon S3 bucket that contains the stored AMI object.</p>
   */
  inline const Aws::String& GetBucket() const { return m_bucket; }
  inline bool BucketHasBeenSet() const { return m_bucketHasBeenSet; }
  template <typename BucketT = Aws::String>
  void SetBucket(BucketT&& value) {
    m_bucketHasBeenSet = true;
    m_bucket = std::forward<BucketT>(value);
  }
  template <typename BucketT = Aws::String>
  CreateRestoreImageTaskRequest& WithBucket(BucketT&& value) {
    SetBucket(std::forward<BucketT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the stored AMI object in the bucket.</p>
   */
  inline const Aws::String& GetObjectKey() const { return m_objectKey; }
  inline bool ObjectKeyHasBeenSet() const { return m_objectKeyHasBeenSet; }
  template <typename ObjectKeyT = Aws::String>
  void SetObjectKey(ObjectKeyT&& value) {
    m_objectKeyHasBeenSet = true;
    m_objectKey = std::forward<ObjectKeyT>(value);
  }
  template <typename ObjectKeyT = Aws::String>
  CreateRestoreImageTaskRequest& WithObjectKey(ObjectKeyT&& value) {
    SetObjectKey(std::forward<ObjectKeyT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name for the restored AMI. The name must be unique for AMIs in the Region
   * for this account. If you do not provide a name, the new AMI gets the same name
   * as the original AMI.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  CreateRestoreImageTaskRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The tags to apply to the AMI and snapshots on restoration. You can tag the
   * AMI, the snapshots, or both.</p> <ul> <li> <p>To tag the AMI, the value for
   * <code>ResourceType</code> must be <code>image</code>.</p> </li> <li> <p>To tag
   * the snapshots, the value for <code>ResourceType</code> must be
   * <code>snapshot</code>. The same tag is applied to all of the snapshots that are
   * created.</p> </li> </ul>
   */
  inline const Aws::Vector<TagSpecification>& GetTagSpecifications() const { return m_tagSpecifications; }
  inline bool TagSpecificationsHasBeenSet() const { return m_tagSpecificationsHasBeenSet; }
  template <typename TagSpecificationsT = Aws::Vector<TagSpecification>>
  void SetTagSpecifications(TagSpecificationsT&& value) {
    m_tagSpecificationsHasBeenSet = true;
    m_tagSpecifications = std::forward<TagSpecificationsT>(value);
  }
  template <typename TagSpecificationsT = Aws::Vector<TagSpecification>>
  CreateRestoreImageTaskRequest& WithTagSpecifications(TagSpecificationsT&& value) {
    SetTagSpecifications(std::forward<TagSpecificationsT>(value));
    return *this;
  }
  template <typename TagSpecificationsT = TagSpecification>
  CreateRestoreImageTaskRequest& AddTagSpecifications(TagSpecificationsT&& value) {
    m_tagSpecificationsHasBeenSet = true;
    m_tagSpecifications.emplace_back(std::forward<TagSpecificationsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Checks whether you have the required permissions for the action, without
   * actually making the request, and provides an error response. If you have the
   * required permissions, the error response is <code>DryRunOperation</code>.
   * Otherwise, it is <code>UnauthorizedOperation</code>.</p>
   */
  inline bool GetDryRun() const { return m_dryRun; }
  inline bool DryRunHasBeenSet() const { return m_dryRunHasBeenSet; }
  inline void SetDryRun(bool value) {
    m_dryRunHasBeenSet = true;
    m_dryRun = value;
  }
  inline CreateRestoreImageTaskRequest& WithDryRun(bool value) {
    SetDryRun(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_bucket;

  Aws::String m_objectKey;

  Aws::String m_name;

  Aws::Vector<TagSpecification> m_tagSpecifications;

  bool m_dryRun{false};
  bool m_bucketHasBeenSet = false;
  bool m_objectKeyHasBeenSet = false;
  bool m_nameHasBeenSet = false;
  bool m_tagSpecificationsHasBeenSet = false;
  bool m_dryRunHasBeenSet = false;
};

}  // namespace Model
}  // namespace EC2
}  // namespace Aws
